/*    Keyboard Driver for Advanced Docking Station      */
/*        3/27/2002   www.PiehlSystems.com              */

/* See kbreadme.txt file for a description of this code */

/* Define onHand buttons & serial port constants */
#define  Key_file		0x0001
#define  Key_enter		0x0008		
#define  Key_el			0x0004		
#define  Key_menu		0x0002		
#define  KB_COMDEV   2    //RS-232
#define  KB_BAUD     5    //9600 bps
#define  KB_PULSE    1    //pulse width (IR only)
#define  KB_TIMOUT   1    //timeout = 1 sec.

/* Variables handled internally by driver */
int KBEnabled=0;          //KEYBOARD.INI-present indicator
int KBConnected=0;        //Connection indicator
int SMode=0;              //Shift/Ctrl/Alt hold value
int RptKey=0;             //Repeating key value
int CapsLocked=0;         //Caps lock indicator
int EndFlag=0;            //Program termination flag
int DrawFlag=0;           //Screen re-draw flag
int ButtonHold=0;         //onHand button hold indicator
int LastCode=0;           //Last key code pressed
char KeyMapNormal[256];   //Normal key mappings
char KeyMapShifted[256];  //Shifted key mappings
#define EventTimeout  60  //Run events after 60 sec. idle

/* ASCII key codes */
#define KB_Tab         9  // ASCII code for tab    (Ctrl-I)
#define KB_Enter      13  // ASCII code for enter  (Ctrl-M)
#define KB_Bksp        8  // ASCII code for backspace (Ctrl-H)
#define KB_Space      32  // ASCII code for space
#define KB_Del       127  // ASCII code for delete

/* Miscellaneous Keys */
#define KB_Ins       128  // Insert key
#define KB_CapsLock  129  // Alt-A or Caps Lock
#define KB_Up        133  // Alt-E or Up key
#define KB_Down      132  // Alt-D or Down key
#define KB_Left      147  // Alt-S or Left key
#define KB_Right     134  // Alt-F or Right key
#define KB_PgUp      151  // Alt-W or Page Up key
#define KB_PgDn      152  // Alt-X or Page Down key
#define KB_Home      146  // Alt-R or Home key
#define KB_End       150  // Alt-V or End key
#define KB_Esc       154  // Alt-Z or Esc key
#define KB_Shift     252  // Shift-in
#define KB_Ctrl      253  // Ctrl-in
#define KB_Alt       254  // Alt-in
#define KB_Release   255  // Shift/Ctrl/Alt-release

/* Identification arrays for named keys (3-significant chars) */
#define KBNameCount 20
char KBNameChar[3*KBNameCount]
  ="TABENTESCBKSSPADELINSCAPSHICTRALTRELUPCDOWLEFRIGPGUPGDHOMEND";
char KBNameCode[KBNameCount]={KB_Tab, KB_Enter, KB_Esc, KB_Bksp,
         KB_Space, KB_Del, KB_Ins, KB_CapsLock, KB_Shift, KB_Ctrl,
         KB_Alt, KB_Release, KB_Up, KB_Down, KB_Left, KB_Right,
         KB_PgUp, KB_PgDn, KB_Home, KB_End};


/* Terminate keyboard connection */

void KBTerminate(void)
{ if (KBEnabled==0) return;
  bi_comclose();         //Close port 
  bi_cominit(0,0,0,0);   //Reset settings
  setSaverCounter();     //Postpone the power-saver
  KBConnected=0;         //Clear connection flag
  SMode=0;
  RptKey=0;
  CapsLocked=0;
  LastCode=0;
}


/* Initialize Keyboard Settings */

void KBInit(void)
{ int i,j,k;      // Counters
  int F;          // File handle
  char C;         // Character code
  char buff[50];  // File I/O buffer
  int bufflen=50; // Buffer size
  char str[6];    // Configuration string
  int strlen=0;   // String length

  /* Reset variables */
  KBEnabled=1;
  KBTerminate();  //Perform cleanup
  KBEnabled=0;
  ButtonHold=0;
  for (i=0; i<=255; ++i)  //Reset key mapping arrays
  { KeyMapNormal[i]=0;
    KeyMapShifted[i]=0;
  }
  
  /* Get keyboard.ini data */
  F=dos_fopen("B:\\SYSTEM\\KEYBOARD.INI",0);  //Open INI file       
  if (F<0) return;                //Exit, if no file
  while (bufflen==50)             //Read data blocks
  { bufflen=dos_fread(F,buff,bufflen);
    for (i=0; i<bufflen; ++i)
    { C=buff[i];
      if (C>32 && C<128)          //Append characters 
      { if (strlen<6) str[strlen++]=C;
      }
      else if (strlen>=4 && str[2]=='=') //Assignment
      { for (j=0; j<6; ++j)      
        { if (j>=strlen)
            str[j]=0;             //Zero out unused bytes
          else if (str[j]>='a' && str[j]<='z' && (j<2 || strlen>5))
            str[j]-=32;           //Set to UCase
        }
        k=0;                      //Reset values
        C=str[0];
        if (C>='0' && C<='9') k+=16*(C-'0');
        if (C>='A' && C<='F') k+=16*(C-'A'+10);
        C=str[1];
        if (C>='0' && C<='9') k+=(C-'0');
        if (C>='A' && C<='F') k+=(C-'A'+10);
        if (strlen<=5)            //For character keys...
        { KeyMapNormal[k]=str[3];     //Store normal key code
          KeyMapShifted[k]=str[4];    //Store shifted key code
        }
        else                      //For named keys...
        { C=0;                    //Reset character code
          for (j=0; j<KBNameCount; ++j)      //Locate name
          { if (  str[3]==KBNameChar[3*j] 
               && str[4]==KBNameChar[3*j+1]
               && str[5]==KBNameChar[3*j+2])
            { C=KBNameCode[j];    //Get corresponding code
              break;
            }
          }
          KeyMapNormal[k]=C;      //Store named key code
          KeyMapShifted[k]=C; 
        }
        strlen=0;
      }
      else
      { strlen=0;
      }
    }
  }
  i=dos_fclose(F);    // Close the keyboard.ini file
  if (i<0) return;

  /* Finish keyboard initialization */
  KBEnabled=1;        //Enable keyboard use

}


/* Get ASCII character code (0=none pressed)         */
/* If return value=0, then no key has been pressed   */
/* Keys between 32 and 126 are printable ASCII codes */

/* Keyboad modes are defined by the following bits:  */
#define KB_Scan       0x0001  // Scan the keyboard without checking events
#define KB_NoRptCrsr  0x0002  // Block repeat of cursor keys
#define KB_NoRptOther 0x0004  // Block repeat of other keys
#define KB_ImmCrsr    0x0008  // Immediate repeat of cursor keys
#define KB_ImmOther   0x0010  // Immediate repeat of other keys

char KBGetKey(int KeyMode)
{ char buff[10];                     // Serial I/O buffer
  int Inkey;                         // Key code value
  int ret;                           // Return value from onHand event

  /* Handle onHand button input */
  ret=0;                             //Reset return value
  if ((KeyMode & KB_Scan)==0 && KBConnected==0)
  { pSystemCtl->EventManager->eventReq  |=  EVENT_SEC;
    Inkey  =  Keyin(0x00F0,-1);      //Wait for button
    if ((Inkey&Bkey_SH)==0) ret = EventCall(Inkey);  //Check for events
  }
  else
  { pSystemCtl->EventManager->eventReq  &=  ~EVENT_SEC;
    Inkey=bi_scanbtn();              //Scan buttons
    bi_clrbtn();                   //Clear the buffer
    if (Inkey&Key_el) bi_el(5);      //Set EL if pressed (5 seconds)
    if (Inkey&Key_file) ret=EventCall(Inkey);  //Handle Filer/Time
    if (Inkey & (Key_menu|Key_enter))  //For MENU/ENTER...
    { if (ButtonHold) Inkey=0; else ButtonHold=1;  //Don't repeat button
    }
    else
    { ButtonHold=0;
    }
  } 
  if(ret==1)
  {  DrawFlag=1;                     //Signal screen drawing
     ret=EventExec();                //Handle event
  }
  if (ret==-1)  EndFlag=1;           //Signal end-of-program
  
  /* Translate internal onHand buttons to key codes */
  if (Inkey&0x00FF)
  { if (Inkey&Key_menu)  return KB_Esc;    //Let MENU button = ESC key
    if (Inkey&Key_enter) return KB_Enter;  //Return ENTER key
    if (Inkey&Bkey_lf)   return KB_Left;   //Return cursor directions
    if (Inkey&Bkey_rg)   return KB_Right;
    if (Inkey&Bkey_dw)   return KB_Down;
    if (Inkey&Bkey_up)   return KB_Up;
  }
  if (KBEnabled==0) return 0;  //Return when keyboard not available

  /* Open and close serial connection as needed */
  if (KBConnected==0)          //If not connected...
  { if ((Inkey&Bkey_SH)==0) return 0; //Exit, if no detection
    KBTerminate(); 
    ret=bi_cominit(KB_COMDEV, KB_BAUD,KB_PULSE,KB_TIMOUT);
    ret=bi_comopen();          //Open com port
    ret=bi_comclr();           //Clear buffer
    KBConnected=1;             //Set connection flag
  }	
  else if ((Inkey&Bkey_SH)==0) //If connection lost...
  { KBTerminate();             //Terminate serial I/O
    return 0;
  } 

  /* Get key code & handle key repeat */
  buff[0]=0;                   //Reset the buffer
  ret=-1;
  if ((KeyMode&KB_Scan)==0)    //For event handling...
  { if (RptKey<0)              //If repeat delay...
    { RptKey=-RptKey;          //Set repeat for next pass
    }
    else if (RptKey)           //For repeat...
    { if (bi_comrnum()==0) return RptKey;
    }
    ret=bi_comread(buff);      //Read key or sleep until pressed
  }
  else if(bi_comrnum())        //otherwise, if key waiting...
  { ret=bi_comread(buff);      //Read code
  }
  if (ret==-1) return 0;       //Exit if timed out
  ret=buff[0];                 //Get code from buffer
  RptKey=0;                    //Reset key repeat after any code
  
  /* Handle key code */
  if (SMode==KB_Shift)         //If shifted...
    Inkey=KeyMapShifted[ret];  //Get shifted code value
  else
    Inkey=KeyMapNormal[ret];   //Get normal code value
  if (Inkey==0 && LastCode==ret) SMode=0;  //Remove shift flag
  LastCode=ret;                //Store last code received
  if (Inkey==0) return 0;      //Ignore unknown keys

  /* Handle Ctrl, Alt, and Caps Lock */
  setSaverCounter();           //Postpone the power-saver
  if ((Inkey>='a' && Inkey<='z') || (Inkey>='A' && Inkey<='Z'))
  { if (SMode==KB_Ctrl || SMode==KB_Alt)    //Handle Ctrl and Alt keys
    { if (Inkey>='a') Inkey-=32;            //UCase
      if (SMode==KB_Ctrl) Inkey-=64; else Inkey+=64;
    }
    else if (CapsLocked)       //If caps lock is on...
    { Inkey ^= 32;               //Swap upper/lower case
    }
  }
  else if (  Inkey==KB_Shift || Inkey==KB_Ctrl 
          || Inkey==KB_Alt   || Inkey==KB_Release )
  { SMode=Inkey;               //Remember shift/ctrl/alt key value
    return 0;
  }
  if (Inkey==KB_CapsLock) //Toggle caps lock
  { CapsLocked=((CapsLocked ^ 1) & 1);
    return 0;
  }

  /* Prepare for immediate/delayed key repeat */
  if (Inkey==KB_Left || Inkey==KB_Right || Inkey==KB_Up || Inkey==KB_Down)
  {  if ((KeyMode & KB_NoRptCrsr)==0) RptKey=-Inkey;
     if (KeyMode & KB_ImmCrsr) RptKey=Inkey;
  }
  else
  {  if ((KeyMode & KB_NoRptOther)==0) RptKey=-Inkey;
     if (KeyMode & KB_ImmOther) RptKey=Inkey;
  }
  return Inkey;                //Return resulting key code
}
